#!/bin/sh
set -e
# Mark as not-for-autoremoval those kernel packages that are:
#  - the currently booted version
#  - the kernel version we've been called for
#  - the latest kernel version (as determined by debian version number)
#  - the second-latest kernel version
#
# In the common case this results in two kernels saved (booted into the
# second-latest kernel, we install the latest kernel in an upgrade), but
# can save up to four. Kernel refers here to a distinct release, which can
# potentially be installed in multiple flavours counting as one kernel.

eval $(apt-config shell APT_CONF_D Dir::Etc::parts/d)
test -n "${APT_CONF_D}" || APT_CONF_D="/etc/apt/apt.conf.d"
config_file="${APT_CONF_D}/01autoremove-kernels"

eval $(apt-config shell DPKG Dir::bin::dpkg/f)
test -n "$DPKG" || DPKG="/usr/bin/dpkg"

list="$("${DPKG}" -l | awk '/^[ih][^nc][ ]+(linux|kfreebsd|gnumach)-image-[0-9]+\./ && $2 !~ /-dbg(:.*)?$/ && $2 !~ /-dbgsym(:.*)?$/ { print $2,$3; }' \
   | sed -e 's#^\(linux\|kfreebsd\|gnumach\)-image-##' -e 's#:[^:]\+ # #')"
debverlist="$(echo "$list" | cut -d' ' -f 2 | sort --unique --reverse --version-sort)"

if [ -n "$1" ]; then
	installed_version="$(echo "$list" | awk "\$1 == \"$1\" { print \$2;exit; }")"
fi
unamer="$(uname -r | tr '[A-Z]' '[a-z]')"
if [ -n "$unamer" ]; then
	running_version="$(echo "$list" | awk "\$1 == \"$unamer\" { print \$2;exit; }")"
fi
# ignore the currently running version if attempting a reproducible build
if [ -n "${SOURCE_DATE_EPOCH}" ]; then
	unamer=""
	running_version=""
fi
latest_version="$(echo "$debverlist" | sed -n 1p)"
previous_version="$(echo "$debverlist" | sed -n 2p)"

debkernels="$(echo "$latest_version
$installed_version
$running_version
$previous_version" | sort -u | sed -e '/^$/ d')"
kernels="$( (echo "$1
$unamer"; for deb in $debkernels; do echo "$list" | awk "\$2 == \"$deb\" { print \$1; }"; done; ) \
   | sed -e 's#\([\.\+]\)#\\\1#g' -e '/^$/ d' | sort -u)"

generateconfig() {
	cat <<EOF
// DO NOT EDIT! File autogenerated by $0
APT::NeverAutoRemove
{
EOF
	for package in $(apt-config dump --no-empty --format '%v%n' 'APT::VersionedKernelPackages'); do
		for kernel in $kernels; do
			echo "   \"^${package}-${kernel}$\";"
		done
	done
	echo '};'
	if [ "${APT_AUTO_REMOVAL_KERNELS_DEBUG:-true}" = 'true' ]; then
		cat <<EOF
/* Debug information:
# dpkg list:
$(dpkg -l | grep '\(linux\|kfreebsd\|gnumach\)-image-')
# list of installed kernel packages:
$list
# list of different kernel versions:
$debverlist
# Installing kernel: $installed_version ($1)
# Running kernel: ${running_version:-ignored} (${unamer:-ignored})
# Last kernel: $latest_version
# Previous kernel: $previous_version
# Kernel versions list to keep:
$debkernels
# Kernel packages (version part) to protect:
$kernels
*/
EOF
	fi
}
generateconfig "$@" > "${config_file}.dpkg-new"
mv -f "${config_file}.dpkg-new" "$config_file"
chmod 444 "$config_file"
