/*
 * $Id: arch_slirp.c,v 1.10 2011-01-26 13:16:18 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef INCLUDE

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <unistd.h>

#include "glue-io.h"

#endif /* INCLUDE */

#ifdef STATE

struct {
	int slirp_fd;
	pid_t slirp_pid;
	pid_t grep_pid;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

#define END	0300
#define ESC	0333
#define ESC_END	0334
#define ESC_ESC	0335

/** send an ip packet to the slip line
 *
 * packet will be encoded and written to the line
 * must already be a bare ip packet, without eth frame
 * called by ip_recv
 */
static int
NAME_(send)(struct cpssp *cpssp, const unsigned char *ip_buf, int ip_len)
{
	unsigned char buf[ETHERMTU*2];
	int i;
	int len;
	int ret;

	/* escape some characters */
	len = 0;
	for (i = 0; i < ip_len; i++) {
		switch (ip_buf[i]) {
		case END:
			buf[len++] = ESC;
			buf[len++] = ESC_END;
			break;
		case ESC:
			buf[len++] = ESC;
			buf[len++] = ESC_ESC;
			break;
		default:
			buf[len++] = ip_buf[i];
			break;
		}
	}
	/* send packet to slirp */
	buf[len++] = END;

	ret = write(cpssp->NAME.slirp_fd, buf, len);
	assert(ret == len); /* FIXME */

	return 0;
}

/** receive ip packet from slip line
 *
 * passes the decoded packet(s) to ip_send
 */
static void
NAME_(interrupt2)(struct cpssp *cpssp, unsigned char *buf, int len)
{
	/* space is reserved for whole packet, including header+crc */
	static unsigned char packet[0x10000 + 2];
	static unsigned int packet_len = 0;
	static int esc = 0;
	int i;

	/* parse them to form an ip packet */
	for (i = 0; i < len; i++) {
		switch (buf[i]) {
		case END:
			if (sizeof(struct ip) < packet_len
			 && packet_len <= 0x10000) {
				/* We have a complete packet now. */
				NAME_(recv)(cpssp, packet, packet_len);
			}
			packet_len = 0;
			break;
		case ESC:
			esc = 1;
			break;
		case ESC_END:
			packet[packet_len++] = esc ? END : ESC_END;
			esc = 0;
			break;
		case ESC_ESC:
			packet[packet_len++] = esc ? ESC : ESC_ESC;
			esc = 0;
			break;
		default:
			packet[packet_len++] = buf[i];
			break;
		}
		if (0x10000 < packet_len) {
			packet_len = 0x10000 + 1;
		}
	}
}

/** receive data from slip line
 *
 * called whenever data is ready.
 * passes data to slip_recv
 */
static void
NAME_(interrupt)(int fd, void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	unsigned char buf[4096];
	int len;

	for (;;) {
		do {
			len = read(cpssp->NAME.slirp_fd, buf, sizeof(buf));
		} while (len < 0 && errno == EINTR);
		if (len < 0 && errno == EAGAIN) {
			break;
		}
		if (len < 0) {
			perror("read");
		} else if (len == 0) {
			fprintf(stderr, "read: len = 0\n");
		}
		assert(0 < len);

		NAME_(interrupt2)(cpssp, buf, len);
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	int errpipe[2];
	int iopipe[2];
	int ret;
	pid_t child_pid;

	ret = pipe(errpipe);
	assert(0 <= ret);

	/* fork grep only output real error messages */
	switch (child_pid = fork()) {
	case -1:
		assert(0); /* Mustn't happen. */
		/*NOTREACHED*/

	case 0:
		ret = dup2(errpipe[0], 0);	/* read from pipe */
		assert(0 <= ret);
		ret = dup2(2, 1);		/* output goes to stderr */
		assert(0 <= ret);
		ret = close(errpipe[0]);
		assert(0 <= ret);
		ret = close(errpipe[1]);
		assert(0 <= ret);
		execlp("grep", "grep", "Error", (char *) 0);
		perror("exec");
		_exit(1);

	default:
		cpssp->NAME.grep_pid = child_pid;
		break;
	}

	ret = socketpair(PF_UNIX, SOCK_STREAM, 0, iopipe);
	assert(0 <= ret);

	switch (child_pid = fork()) {
	case -1:
		assert(0); /* Shouldn't happen. */
		/*NOTREACHED*/

	case 0: /* child, executing slirp */
		ret = dup2(iopipe[0], 0);
		assert(0 <= ret);
		ret = close(1);
		assert(0 <= ret);
		ret = open("/dev/null", O_WRONLY);
		assert(ret == 1);
		ret = dup2(errpipe[1], 2);	/* stderr goes to above pipe */
		assert(0 <= ret);
		ret = close(iopipe[0]);
		assert(0 <= ret);
		ret = close(iopipe[1]);
		assert(0 <= ret);
		ret = close(errpipe[0]);
		assert(0 <= ret);
		ret = close(errpipe[1]);
		assert(0 <= ret);
		execlp("slirp", "slirp", "mtu 1486", (char *) 0);
		/* mtu = 1500 - sizeof(ethhdr) */
		fprintf(stderr, "Error: slirp: %s.", strerror(errno));
		_exit(1);

	default:
		cpssp->NAME.slirp_pid = child_pid;
		break;
	}

	ret = close(iopipe[0]);
	assert(0 <= ret);

	ret = close(errpipe[0]);
	assert(0 <= ret);
	ret = close(errpipe[1]);
	assert(0 <= ret);

	cpssp->NAME.slirp_fd = iopipe[1];

	io_register(cpssp->NAME.slirp_fd, cpssp, NAME_(interrupt));
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
	int ret;

	io_unregister(cpssp->NAME.slirp_fd);

	ret = close(cpssp->NAME.slirp_fd);
	assert(0 <= ret);

	ret = kill(cpssp->NAME.slirp_pid, 15);
	assert(0 <= ret);

	ret = waitpid(cpssp->NAME.slirp_pid, NULL, 0);
	assert(ret == cpssp->NAME.slirp_pid);

	ret = kill(cpssp->NAME.grep_pid, 15);
	assert(0 <= ret);

	ret = waitpid(cpssp->NAME.grep_pid, NULL, 0);
	assert(ret == cpssp->NAME.grep_pid);
}

#endif /* BEHAVIOR */
