/* -*- Mode: C; c-basic-offset:4 ; indent-tabs-mode:nil -*- */
/*
 * Copyright (c) 2004-2007 The Trustees of Indiana University and Indiana
 *                         University Research and Technology
 *                         Corporation.  All rights reserved.
 * Copyright (c) 2004-2020 The University of Tennessee and The University
 *                         of Tennessee Research Foundation.  All rights
 *                         reserved.
 * Copyright (c) 2004-2005 High Performance Computing Center Stuttgart,
 *                         University of Stuttgart.  All rights reserved.
 * Copyright (c) 2004-2005 The Regents of the University of California.
 *                         All rights reserved.
 * Copyright (c) 2013      Los Alamos National Security, LLC.  All rights
 *                         reserved.
 * Copyright (c) 2015      Research Organization for Information Science
 *                         and Technology (RIST). All rights reserved.
 * Copyright (c) 2016-2017 IBM Corporation. All rights reserved.
 * Copyright (c) 2021-2022 Triad National Security, LLC. All rights
 *                         reserved.
 * $COPYRIGHT$
 *
 * Additional copyrights may follow
 *
 * $HEADER$
 */

#include "ompi_config.h"

#include "ompi/mpi/c/bindings.h"
#include "ompi/runtime/params.h"
#include "ompi/communicator/communicator.h"
#include "ompi/errhandler/errhandler.h"
#include "ompi/info/info.h"
#include "opal/util/string_copy.h"
#include <stdlib.h>
#include <string.h>

#if OMPI_BUILD_MPI_PROFILING
#if OPAL_HAVE_WEAK_SYMBOLS
#pragma weak MPI_Info_get_string = PMPI_Info_get_string
#endif
#define MPI_Info_get_string PMPI_Info_get_string
#endif

static const char FUNC_NAME[] = "MPI_Info_get_string";

/**
 *   MPI_Info_get_string - Get a (key, value) pair from an 'MPI_Info' object
 *
 *   @param info info object (handle)
 *   @param key null-terminated character string of the index key
 *   @param buflen maximum length of 'value' (integer)
 *   @param value null-terminated character string of the value
 *   @param flag true (1) if 'key' defined on 'info', false (0) if not
 *               (logical)
 *
 *   @retval MPI_SUCCESS
 *   @retval MPI_ERR_ARG
 *   @retval MPI_ERR_INFO
 *   @retval MPI_ERR_INFO_KEY
 *   @retval MPI_ERR_INFO_VALUE
 *
 */
int MPI_Info_get_string(MPI_Info info, const char *key, int *buflen,
                        char *value, int *flag)
{
    int err;
    int key_length;
    opal_cstring_t *info_str;

    /*
     * Simple function. All we need to do is search for the value
     * having the "key" associated with it and then populate the
     * necessary structures.
     */
    if (MPI_PARAM_CHECK) {
        if (NULL == info || MPI_INFO_NULL == info ||
            ompi_info_is_freed(info)) {
            return OMPI_ERRHANDLER_NOHANDLE_INVOKE(MPI_ERR_INFO,
                                          FUNC_NAME);
        }

        key_length = (key) ? (int)strlen (key) : 0;
        if ((NULL == key) || (0 == key_length) ||
            (MPI_MAX_INFO_KEY <= key_length)) {
            return OMPI_ERRHANDLER_NOHANDLE_INVOKE(MPI_ERR_INFO_KEY,
                                          FUNC_NAME);
        }
        if (NULL == buflen) {
            return OMPI_ERRHANDLER_NOHANDLE_INVOKE(MPI_ERR_ARG,
                                          FUNC_NAME);
        }
        if ((NULL == value) && *buflen) {
            return OMPI_ERRHANDLER_NOHANDLE_INVOKE(MPI_ERR_INFO_VALUE,
                                          FUNC_NAME);
        }
        if (NULL == flag) {
            return OMPI_ERRHANDLER_NOHANDLE_INVOKE(MPI_ERR_ARG,
                                          FUNC_NAME);
        }
    }

    if (0 == *buflen) {
        err = ompi_info_get_valuelen(info, key, buflen, flag);
        if (1 == *flag) {
            *buflen += 1; /* add on for the \0, see MPI 4.0 Standard */
        }
    } else {
        err = ompi_info_get(info, key, &info_str, flag);
        if (*flag) {
            opal_string_copy(value, info_str->string, *buflen);
            *buflen = info_str->length + 1; /* add on for the \0, see MPI 4.0 Standard */
            OBJ_RELEASE(info_str);
        }
    }

    OMPI_ERRHANDLER_NOHANDLE_RETURN(err, err, FUNC_NAME);
}
